//
//  $Id: AHKMMainWindowController_IO.m 98 2009-06-12 17:01:44Z fujidana $
//  Copyright (c) 2006-2009 Fujidana All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
// IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
// INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
// NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
// THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//

#import "AHKMMainWindowController.h"
#import "AHKPhone_Mail.h"
#import "AHKPasswordWindowController.h"
#import "AHKProgressWindowController.h"
#import "AHKMail.h"


@interface AHKMMainWindowController (IO_Private)
#pragma mark for communitation
- (void)receiveMailsSheetDidEnd:(NSAlert *)sheet returnCode:(int)returnCode contextInfo:(void *)contextInfo;

#pragma mark other methods

- (BOOL)readDatabaseFromFile:(NSString *)path;
- (BOOL)writeDatabaseToFile:(NSString *)path withFormerFileStored:(BOOL)storedFlag extensionHidden:(BOOL)hiddenFlag;
- (NSString *)pathForDataFile;

- (void)writeMboxOfMails:(NSArray *)mails toDirectory:(NSString *)directory;
- (void)writeMailsAsRFC822Message:(NSArray *)mails toDirectory:(NSString *)directory;
- (void)writeVMessageOfMails:(NSArray *)mails toDirectory:(NSString *)directory;

- (void)updateReceivedDateWithDate:(NSDate *)date;
- (void)showAlertWithError:(NSError *)error;

@end

#pragma mark -

@implementation AHKMMainWindowController (IO)


#pragma mark IBActions and their delegation methods

#pragma mark methods to import and export
- (IBAction)exportSelection:(id)sender
{
	NSOpenPanel *openPanel = [NSOpenPanel openPanel];
	[openPanel setAllowsMultipleSelection:NO];
	[openPanel setCanChooseDirectories:YES];
	[openPanel setCanChooseFiles:NO];
	[openPanel setCanCreateDirectories:YES];
	[openPanel setAccessoryView:fileFormatView];
	[openPanel setTitle:NSLocalizedStringFromTable(@"Export", @"AHKMLocalizable", @"exportSheet.title.export")];
	[openPanel setPrompt:NSLocalizedStringFromTable(@"Export", @"AHKMLocalizable", @"exportSheet.prompt.export")];
	
	NSUserDefaults *userDefaults = [NSUserDefaults standardUserDefaults];
	[fileFormatPopUpButton selectItemAtIndex:[userDefaults integerForKey:AHKMFileFormatDefaultsKey]];
	[phoneNumberTextField setStringValue:[userDefaults stringForKey:AHKMPhoneNumberDefaultsKey]];
	
	[openPanel beginSheetForDirectory:nil
								 file:nil
								types:nil
					   modalForWindow:[self window]
						modalDelegate:self
					   didEndSelector:@selector(exportSelectionOpenPanelDidEnd:returnCode:contextInfo:)
						  contextInfo:nil];
}

- (void)exportSelectionOpenPanelDidEnd:(NSOpenPanel *)openPanel returnCode:(int)returnCode contextInfo:(void *)contextInfo
{
	if (returnCode == NSOKButton)
	{
		AHKMFileFormat format = [fileFormatPopUpButton indexOfSelectedItem];
		NSString *phoneNumber = [phoneNumberTextField stringValue];
		
		NSUserDefaults *userDefaults = [NSUserDefaults standardUserDefaults];
		[userDefaults setObject:phoneNumber forKey:AHKMPhoneNumberDefaultsKey];
		[userDefaults setInteger:format forKey:AHKMFileFormatDefaultsKey];
		
		NSString *directoryPath = [openPanel filename];
		NSArray *mails = [mailsController selectedObjects];
		
		switch (format)
		{
			case AHKMMboxFileFormat:
				[self writeMboxOfMails:mails toDirectory:directoryPath];
				break;
			case AHKMRFC822MessageFileFormat:
				[self writeMailsAsRFC822Message:mails toDirectory:directoryPath];
				break;
			case AHKMVMessageFileFormat:
				[self writeVMessageOfMails:mails toDirectory:directoryPath];
				break;
		}
	}
}

#pragma mark for save and revert
- (IBAction)backupDatabase:(id)sender
{
	NSString       *filenameMesssage = NSLocalizedStringFromTable(@"Kyopon Backup - %@", @"AHKMLocalizable", @"backupDatabase.filenameMessage");
	NSUserDefaults *userDefaults     = [NSUserDefaults standardUserDefaults];
	NSString       *shortDateFormat  = [userDefaults stringForKey:NSShortDateFormatString];
	
	NSString       *filenameDate     = [[NSCalendarDate calendarDate] descriptionWithCalendarFormat:shortDateFormat];
	
	NSSavePanel    *savePanel        = [NSSavePanel savePanel];
	
	[savePanel setRequiredFileType:@"ahkmbu"];
	//[savePanel setExtensionHidden:YES];
	[savePanel setCanSelectHiddenExtension:YES];
	[savePanel beginSheetForDirectory:nil
								 file:[NSString stringWithFormat:filenameMesssage, filenameDate]
					   modalForWindow:[self window]
						modalDelegate:self
					   didEndSelector:@selector(backupDatabaseSavePanelDidEnd:returnCode:contextInfo:)
						  contextInfo:nil];
}

- (void)backupDatabaseSavePanelDidEnd:(NSSavePanel *)savePanel returnCode:(int)returnCode contextInfo:(void *)contextInfo
{
	if (returnCode == NSOKButton)
	{
		[self writeDatabaseToFile:[savePanel filename]
			 withFormerFileStored:NO
				  extensionHidden:[savePanel isExtensionHidden]];
	}
}

- (IBAction)revertFromDatabaseBackup:(id)sender
{
	NSOpenPanel *openPanel = [NSOpenPanel openPanel];
	[openPanel setAllowsMultipleSelection:NO];
	[openPanel setCanChooseDirectories:NO];
	[openPanel setCanChooseFiles:YES];
	
	[openPanel beginSheetForDirectory:nil
								 file:nil
								types:[NSArray arrayWithObjects:@"ahkmbu", nil]
					   modalForWindow:[self window]
						modalDelegate:self
					   didEndSelector:@selector(revertFromDatabaseBackupOpenPanelDidEnd:returnCode:contextInfo:)
						  contextInfo:nil];
}

- (void)revertFromDatabaseBackupOpenPanelDidEnd:(NSOpenPanel *)openPanel returnCode:(int)returnCode contextInfo:(void *)contextInfo
{
	if (returnCode == NSOKButton)
	{
		//		[self readDatabaseFromFile:[openPanel filename]];
		[openPanel orderOut:nil];
		[self confirmRevertDatabaseFromFile:[openPanel filename]];
	}
}

#pragma mark for communication with Kyopon
- (IBAction)receiveMails:(id)sender
{
	// if this method is invoked when the text field is under edit, the currently edited value of text field is not synchronized with user defaults. To force synchronizing them, change the first responder and finish editing.
	[[self window] makeFirstResponder:nil];
	
	if ([[self mails] count] > 0)
	{
		NSAlert *alert = [[[NSAlert alloc] init] autorelease];
		[alert addButtonWithTitle:NSLocalizedStringFromTable(@"Receive", @"AHKMLocalizable", @"receiveSheet.button.receive")];
		[alert addButtonWithTitle:NSLocalizedStringFromTable(@"Cancel", @"AHKMLocalizable", @"alertSheet.button.cancel")];
		[alert setMessageText:NSLocalizedStringFromTable(@"Are you sure you want to overwrite the mails on this computer with the data received from your Kyopon?", @"AHKMLocalizable", @"receiveMailsSheet.message")];
		[alert setInformativeText:NSLocalizedStringFromTable(@"Any changes you've made on this computer will be lost.", @"AHKMLocalizable", @"receiveMailsSheet.information")];
		[alert setAlertStyle:NSInformationalAlertStyle];
		[alert beginSheetModalForWindow:[self window]
						  modalDelegate:self
						 didEndSelector:@selector(receiveMailsSheetDidEnd:returnCode:contextInfo:)
							contextInfo:nil];
	}
	else
	{
		[self receiveMailsSheetDidEnd:nil returnCode:NSAlertFirstButtonReturn contextInfo:nil];
	}
}

- (void)receiveMailsSheetDidEnd:(NSAlert *)alert returnCode:(int)returnCode contextInfo:(void *)contextInfo
{
	[[alert window] close];
	
	if(returnCode == NSAlertFirstButtonReturn)
	{
		AHKPhone                    *phone                    = [[[AHKPhone alloc] init] autorelease];
		AHKPasswordWindowController *passwordWindowController = [AHKPasswordWindowController sharedWindowController];
		AHKProgressWindowController *progressWindowController = [AHKProgressWindowController sharedWindowControllerWithParentWindow:[self window]];
		NSString                    *password                 = [passwordWindowController findPassword];
		NSArray                     *array;
		
		if (password == nil)
		{
			[passwordWindowController inputPasswordForWindow:[self window]
											   modalDelegate:self 
										didEndActionSelector:@selector(receiveMails:)];
			return;
		}
		
		NSError *error;
		BOOL flag = [phone readMails:&array password:password progress:progressWindowController error:&error];
		
		// failed to receive mails
		if (flag == NO)
		{
			// Since Mac OS X 10.3 does not support error-presentation and error-recovery architectures, handle the error manually.
			// in case password is incorrect
			if ([[error domain] isEqualToString:AHKErrorDomain] &&
				([error code] == AHKPhoneIncorrectPasswordError ||
				 [error code] == AHKPhoneInapplicablePasswordStringEncodingError || 
				 [error code] == AHKPhoneInapplicablePasswordStringLengthError))
			{
				[passwordWindowController inputPasswordForWindow:[self window]
												   modalDelegate:self
											didEndActionSelector:@selector(receiveMails:)];
			}
			else
			{
				[self showAlertWithError:error];
			}
		}
		// Succeeded in receiving mails
		else
		{
			NSUndoManager	*undoManager	= [[self window] undoManager];
			[self setMails:array];
			[self updateReceivedDateWithDate:[NSDate date]];
			[undoManager setActionName:NSLocalizedStringFromTable(@"Apply Received Data", @"AHKMLocalizable", @"undo.receive")];
		}
	}
}

#pragma mark other public methods

- (void)confirmRevertDatabaseFromFile:(NSString *)filename
{
	NSAlert *alert = [[[NSAlert alloc] init] autorelease];
	[alert setMessageText:NSLocalizedStringFromTable(@"Reverting will cause your current database to be overwirtten", @"AHKMLocalizable", @"revertComfirm.message")];
	[alert setInformativeText:[NSString stringWithFormat:NSLocalizedStringFromTable(@"The backup database %@ will be copied over your existing database.", @"AHKMLocalizable", @"revertComfirm.information"), [filename lastPathComponent]]];
	[alert setAlertStyle:NSCriticalAlertStyle];
	[alert addButtonWithTitle:NSLocalizedStringFromTable(@"OK", @"AHKMLocalizable", @"alertSheet.button.ok")];
	[alert addButtonWithTitle:NSLocalizedStringFromTable(@"Cancel", @"AHKMLocalizable", @"alertSheet.button.cancel")];
	
	int result = [alert runModal];
	
	if (result == NSAlertFirstButtonReturn)
	{
		[self readDatabaseFromFile:filename];
	}
}

- (BOOL)loadDefaultDatabase
{
	return [self readDatabaseFromFile:[self pathForDataFile]];
}

- (BOOL)storeDefaultDatabase
{
	return [self writeDatabaseToFile:[self pathForDataFile] 
				withFormerFileStored:YES
					 extensionHidden:NO];
}

#pragma mark private auxiliary methods

- (BOOL)readDatabaseFromFile:(NSString *)path
{
	NSFileManager *fileManager = [NSFileManager defaultManager];
	BOOL          isDirectory;
	
	if ([fileManager fileExistsAtPath:path isDirectory:&isDirectory] && isDirectory == NO)
	{
		id dataContent = [NSKeyedUnarchiver unarchiveObjectWithFile:path];
		if (dataContent != nil)
		{
			[self setMails:dataContent];
			return YES;
		}
	}
	return NO;
}

- (BOOL)writeDatabaseToFile:(NSString *)path withFormerFileStored:(BOOL)storedFlag extensionHidden:(BOOL)hiddenFlag
{
	NSFileManager *fileManager = [NSFileManager defaultManager];
	
	if (storedFlag == YES)
	{
		// if a data file already exists, rename it by adding tilde at the end of its filename.
		NSString *pathExtension  = [path pathExtension];
		NSString *backupFilename = [[[path stringByDeletingPathExtension] stringByAppendingString:@"~"] stringByAppendingPathExtension:pathExtension];
		
		if ([fileManager fileExistsAtPath:path])
		{
			if ([fileManager fileExistsAtPath:backupFilename])
			{
				[fileManager removeFileAtPath:backupFilename handler:nil];
			}
			[fileManager movePath:path toPath:backupFilename handler:nil];
		}
	}
	
	// store data
	if ([self mails] != nil)
	{
		if ([NSKeyedArchiver archiveRootObject:[self mails] toFile:path])
		{
			// If the data are stored successfully, then set the path extension hidden or not.
			NSDictionary *attributes = [NSDictionary dictionaryWithObject:[NSNumber numberWithBool:hiddenFlag]
																   forKey:NSFileExtensionHidden];
			
			[fileManager changeFileAttributes:attributes
									   atPath:path];
			return YES;
		}
	}
	
	return NO;
	
}

- (NSString *)pathForDataFile
{
	NSFileManager *fileManager = [NSFileManager defaultManager];
	
	//	// NSApplicationSupportDirectory is available in Mac OS X 10.4 and later.
	//	NSArray *array = NSSearchPathForDirectoriesInDomains(NSApplicationSupportDirectory, NSUserDomainMask, YES);
	
	// if "Application Support" doesn't exist, create it.
	NSString *pathForApplicationSupport = [[NSHomeDirectory()
											stringByAppendingPathComponent: @"Library"]
										   stringByAppendingPathComponent:@"Application Support"];
	if ([fileManager fileExistsAtPath:pathForApplicationSupport] == NO)
	{
		[fileManager createDirectoryAtPath:pathForApplicationSupport attributes:nil];
	}
	
	// if "AH-K3001V Utilities" doesn't exist, create it.
	NSString *pathForDataDirectory = [pathForApplicationSupport stringByAppendingPathComponent:@"AH-K3001V Utilities"];
	if ([fileManager fileExistsAtPath:pathForDataDirectory] == NO)
	{
		[fileManager createDirectoryAtPath:pathForDataDirectory attributes:nil];
	}
	
	// return path for the data file
	return [pathForDataDirectory stringByAppendingPathComponent:@"MailData.ahkutil"];
}


- (void)writeMboxOfMails:(NSArray *)mails toDirectory:(NSString *)directory
{
	AHKMail <AHKMailing> *mail;
	NSEnumerator         *enumerator = [mails objectEnumerator];
	NSAutoreleasePool    *pool;
	NSString             *extraHeader;
	
	NSMutableData *lightMailInboxData  = [NSMutableData dataWithCapacity:1024];
	NSMutableData *lightMailOutboxData = [NSMutableData dataWithCapacity:1024];
	NSMutableData *emailInboxData      = [NSMutableData dataWithCapacity:65535];
	NSMutableData *emailOutboxData     = [NSMutableData dataWithCapacity:65535];
	
	while (mail = [enumerator nextObject])
	{
		pool = [[NSAutoreleasePool alloc] init];
		
		if ([mail isMemberOfClass:[AHKLightMail class]])
		{
			extraHeader = [NSString stringWithFormat:@"From %@ %@\r\nX-KYOPON-LIGHT:%d,%d,%d,%s\r\nContent-Type: text/plain; charset=Shift-JIS\r\n",
						   [mail from],
						   [[mail date] descriptionWithCalendarFormat:@"%a %b %e %H:%M:%S %Y" timeZone:nil locale:nil],
						   [mail box],
						   [mail number],
						   [mail status],
						   ([mail isSecured]? "ON": "OFF")];
			
			if ([mail box] == AHKInboxMailBox)
			{
				[lightMailInboxData appendData:[extraHeader dataUsingEncoding:NSShiftJISStringEncoding]];
				[lightMailInboxData appendData:[mail message]];
				[lightMailInboxData appendData:[@"\r\n\r\n" dataUsingEncoding:NSShiftJISStringEncoding]];
			}
			else
			{
				[lightMailOutboxData appendData:[extraHeader dataUsingEncoding:NSShiftJISStringEncoding]];
				[lightMailOutboxData appendData:[mail message]];
				[lightMailOutboxData appendData:[@"\r\n\r\n" dataUsingEncoding:NSShiftJISStringEncoding]];
			}
		}
		else if ([mail isMemberOfClass:[AHKEmail class]])
		{
			extraHeader = [NSString stringWithFormat:@"From %@ %@\r\nX-KYOPON-MAIL:%d,%d,%d,%s\r\n",
						   [mail from],
						   [[mail date] descriptionWithCalendarFormat:@"%a %b %e %H:%M:%S %Y" timeZone:nil locale:nil],
						   [mail box],
						   [mail number],
						   [mail status],
						   ([mail isSecured]? "ON": "OFF")];
			
			if ([mail box] == AHKInboxMailBox)
			{
				[emailInboxData appendData:[extraHeader dataUsingEncoding:NSISO2022JPStringEncoding]];
				[emailInboxData appendData:[mail message]];
			}
			else
			{
				[emailOutboxData appendData:[extraHeader dataUsingEncoding:NSISO2022JPStringEncoding]];
				[emailOutboxData appendData:[mail message]];
			}
		}
		[pool release];
	}
	
	if ([emailInboxData length] != 0)
	{
		[emailInboxData writeToFile:[directory stringByAppendingPathComponent:@"emailInbox.mbox"]
						 atomically:YES];
	}
	if ([emailOutboxData length] != 0)
	{
		[emailOutboxData writeToFile:[directory stringByAppendingPathComponent:@"emailOutbox.mbox"]
						  atomically:YES];
	}
	if ([lightMailInboxData length] != 0)
	{
		[lightMailInboxData writeToFile:[directory stringByAppendingPathComponent:@"lightMailInbox.mbox"]
							 atomically:YES];
	}
	if ([lightMailOutboxData length] != 0)
	{
		[lightMailOutboxData writeToFile:[directory stringByAppendingPathComponent:@"lightMailOutbox.mbox"]
							  atomically:YES];
	}
}

- (void)writeMailsAsRFC822Message:(NSArray *)mails toDirectory:(NSString *)directory
{
	AHKMail	<AHKMailing> *mail;
	NSEnumerator         *enumerator = [mails objectEnumerator];
	NSAutoreleasePool    *pool;
	NSString             *filename;
	
	while (mail = [enumerator nextObject])
	{
		pool     = [[NSAutoreleasePool alloc] init];
		filename = nil;
		
		if ([mail isMemberOfClass:[AHKLightMail class]])
		{
			filename = [NSString stringWithFormat:@"lightmail_%d_%d_%d_%s.eml",
						[mail box],
						[mail number],
						[mail status],
						([mail isSecured]? "ON": "OFF")];
		}
		else if ([mail isMemberOfClass:[AHKEmail class]])
		{
			filename = [NSString stringWithFormat:@"email_%d_%d_%d_%s.eml",
						[mail box],
						[mail number],
						[mail status],
						([mail isSecured]? "ON": "OFF")];
		}
		if (filename)
		{
			[[mail message] writeToFile:[directory stringByAppendingPathComponent:filename]
							 atomically:YES];
		}
		[pool release];
	}
}

- (void)writeVMessageOfMails:(NSArray *)mails toDirectory:(NSString *)directory
{
	AHKMail <AHKMailing> *mail;
	NSEnumerator         *enumerator = [mails objectEnumerator];
	
	NSAutoreleasePool    *pool;
	NSString             *vmgHeader;
	NSString             *vmgFooter  = @"END:VBODY\r\nEND:VENV\r\nEND:VMSG\r\n";
	
	NSMutableData        *inboxData  = [NSMutableData dataWithCapacity:[mails count] * 256];
	NSMutableData        *outboxData = [NSMutableData dataWithCapacity:[mails count] * 256];
	
	while (mail = [enumerator nextObject])
	{
		pool = [[NSAutoreleasePool alloc] init];
		
		NSString *irmcStatus = ([mail status] == 2)? @"READ": @"UNREAD";
		NSString *irmcType   = ([mail isMemberOfClass:[AHKLightMail class]])? @"SMS" : @"INET";
		
		NSString *irmcBox;
		NSString *wx310kMailFolder;
		
		switch ([mail box])
		{
			case AHKSentMailBox:
				irmcBox = @"SENTBOX";
				wx310kMailFolder = @"SENT";
				break;
			case AHKDraftMailBox:
				irmcBox = @"SENTBOX";
				wx310kMailFolder = @"SENT";
				break;
			case AHKInboxMailBox:
				irmcBox = @"INBOX";
				wx310kMailFolder = @"DEFAULT";
				break;
			default:
				irmcBox = @"INBOX";
				wx310kMailFolder = @"DEFAULT";
		}
		
		NSString* wx310kProtect = [mail isSecured]? @"ON": @"OFF";
		
		vmgHeader = [NSString stringWithFormat:@"BEGIN:VMSG\r\nVERSION:1.1\r\nX-IRMC-STATUS:%@\r\nX-IRMC-TYPE:%@\r\nX-IRMC-BOX:%@\r\nX-WX310K-MAILFOLDER:%@\r\nX-WX310K-PROTECT:%@\r\nBEGIN:VENV\r\nBEGIN:VBODY\r\n",
					 irmcStatus,
					 irmcType,
					 irmcBox,
					 wx310kMailFolder,
					 wx310kProtect];
		
		if ([mail box] == AHKDraftMailBox || [mail box] == AHKSentMailBox)
		{
			[outboxData appendData:[vmgHeader dataUsingEncoding:NSASCIIStringEncoding]];
			[outboxData appendData:[mail message]];
			[outboxData appendData:[vmgFooter dataUsingEncoding:NSASCIIStringEncoding]];
		}
		else
		{
			[inboxData appendData:[vmgHeader dataUsingEncoding:NSASCIIStringEncoding]];
			[inboxData appendData:[mail message]];
			[inboxData appendData:[vmgFooter dataUsingEncoding:NSASCIIStringEncoding]];
		}
		[pool release];
	}
	
	if ([inboxData length] != 0)
	{
		[inboxData writeToFile:[directory stringByAppendingPathComponent:@"RM310K.vmg"]
					atomically:YES];
	}
	if ([outboxData length] != 0)
	{
		[outboxData writeToFile:[directory stringByAppendingPathComponent:@"SM310K.vmg"]
					 atomically:YES];
	}
}

// small method to support undoing and redoing of the last sync date
- (void)updateReceivedDateWithDate:(NSDate *)date
{
	NSUndoManager  *undoManager  = [[self window] undoManager];
	NSUserDefaults *userDefaults = [NSUserDefaults standardUserDefaults];
	[undoManager registerUndoWithTarget:self selector:@selector(updateReceivedDateWithDate:) object:[userDefaults objectForKey:AHKMDefaultsLastReceivedDateKey]];
	[userDefaults setObject:date forKey:AHKMDefaultsLastReceivedDateKey];
}

// --- -showAlertWithMessage:exception: was replaced with this method.
// --- Since Mac OS X 10.3 does not support +alertWithError: of NSError, we use this method to generate alert manually.

- (void)showAlertWithError:(NSError *)error
{
	NSAlert *alert = [[[NSAlert alloc] init] autorelease];
	
	NSString *description = [error localizedDescription];
	if (description)
	{
		[alert setMessageText:description];
	}
	//	[alert setInformativeText:<#(NSString *)informativeText#>];
	[alert addButtonWithTitle:NSLocalizedStringFromTable(@"OK", @"AHKMLocalizable", @"alertSheet.button.ok")];
	[alert setAlertStyle:NSCriticalAlertStyle];
	//	[alert setHelpAnchor:@"ahkabhlp1021"];
	[alert setShowsHelp:YES];
	
	[alert beginSheetModalForWindow:[self window]
					  modalDelegate:self
					 didEndSelector:nil
		                contextInfo:nil];
}

@end
